///////////////////////////////////////////////////////////
//		ISO_SYSTEMS.WDL          
//
//    Copyright 2004 Sean Patrick Hannifin
//		Created for 3DGS Isometric Tutorial
//		Version 1.00  -  August 2004
//
///////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////
// Global variables

entity* MainObject;	// Main object of the camera, used in
// the camera_control() functions.
entity* BlueSquare;	// Pointer to blue square

var CamZoom = 300;	// The height of the camera above the
// main object
var CamAngle = 0;		// The angle the camera is viewing the
// object

// These variables are used in the Camera_Control functions
var ZoomSpeed = 2;
var RotateSpeed = 0.3;
var MoveSpeed = 2.5;
// Zoom Limits
var ZoomDownLimit = 200;
var ZoomUpLimit = 1400;
// Map Limits
var XMax = 640;
var XMin = -640;
var YMax = 640;
var YMin = -640;

// Tile Size - in quants, of course
// 64 is equal to 4x4 boxes in WED
var TileSize = 64;
// Grid Dimensions
var GridLength = 20;
var GridWidth = 20;
// Nodes - array length = GridLength * GridWidth
var Nodes[400];

// Build mode determines whether objects are being
// built (0) or deleted (1)
var BuildMode = 0;

// bmap declaration for the mouse
bmap yellowarrow = <yellowarrow.bmp>;

// string holds path texture
string earthtile = <earthtile.bmp>;
// square and delete square files
string square1 = <square.bmp>;
string square2 = <squaredel.bmp>;

// sound for beeping
sound beep_sound = <beep.wav>;

///////////////////////////////////////////////////////////
// The camera

// This action is to be assigned to the main object of the
// camera
action CameraPoint
{
	// Set the entity equal to the MainObject entity pointer
	MainObject = me;
	
	// Make me invisible and passable
	my.invisible = on;
	my.passable = on;
}

// This function initializes the camera
function Camera_Init()
{
	// We have to wait for the level to load, so the engine
	// doesn't think that the point MainObject is empty
	wait(10);
	
	Camera_Control();
	Camera_Control2();
	
	// Keep updating the camera's position
	while(1)
	{	
		// Set the camera's position in space based on
		// the position of MainObject;
		camera.z = CamZoom;
		camera.x = MainObject.x + CamZoom * cos(CamAngle);
		camera.y = MainObject.y + CamZoom * sin(CamAngle);
		
		// Turn the camera towards the MainObject
		vec_set(temp,MainObject.x);
		vec_sub(temp,camera.x);
		vec_to_angle(camera.pan,temp);
		
		// Reset CamAngle if it is at 360 or above
		if (CamAngle > 359)
		{
			CamAngle = 0;		
		}
		wait(1);
	}
}

// This function is called from the Camera_Control functions
// It tests to make sure that MainObject does not go
// 'out of bounds'
function TestLimits()
{
	MainObject.x = max(min(MainObject.x, XMax), XMin);
	MainObject.y = max(min(MainObject.y, Ymax), YMin);
}

// These functions control the movement of the camera and
// of MainObject
function Camera_Control()
{
	while (1)
	{
		while ((key_cuu == 1) && (key_ctrl == 1) && (key_cud != 1))
		{
			CamZoom -= ZoomSpeed;
			if (CamZoom < ZoomDownLimit)
			{
				CamZoom = ZoomDownLimit;
			}
			wait(1);
		}
		while ((key_cud == 1) && (key_ctrl == 1) && (key_cuu != 1))
		{
			CamZoom += ZoomSpeed;
			if (CamZoom > ZoomUpLimit)
			{
				CamZoom = ZoomUpLimit;
			}
			wait(1);
		}
		while ((key_cuu == 1) && (key_ctrl != 1) && (key_cud != 1))
		{
			MainObject.y += sin(camera.pan) * MoveSpeed;
			MainObject.x += cos(camera.pan) * MoveSpeed;
			TestLimits();
			wait(1);
		}
		while ((key_cud == 1) && (key_ctrl != 1) && (key_cuu != 1))
		{
			MainObject.y -= sin(camera.pan) * MoveSpeed;
			MainObject.x -= cos(camera.pan) * MoveSpeed;
			TestLimits();
			wait(1);
		}
		wait(1);
	}
}

function Camera_Control2()
{
	while(1)
	{
		while((key_cul == 1) && (key_ctrl == 1) && (key_cur != 1))
		{
			CamAngle -= RotateSpeed;
			wait(1);
		}
		while((key_cur ==1) && (key_ctrl == 1) && (key_cul != 1))
		{
			CamAngle += RotateSpeed;
			wait(1);
		}
		while((key_cur == 1) && (key_ctrl != 1) && (key_cul != 1))
		{
			MainObject.y -= cos(camera.pan) * MoveSpeed;
			MainObject.x += sin(camera.pan) * MoveSpeed;
			TestLimits();
			wait(1);
		}
		while((key_cul == 1) && (key_ctrl != 1) && (key_cur != 1))
		{
			MainObject.y += cos(camera.pan) * MoveSpeed;
			MainObject.x -= sin(camera.pan) * MoveSpeed;
			TestLimits();
			wait(1);
		}
		wait(1);
	}
}

///////////////////////////////////////////////////////////
// Nodes and Tiles

// This function initializes the nodes
function Node_Init()
{
	var InitLoop = 0;
	
	while (InitLoop < (GridLength * GridWidth))
	{
		Nodes[InitLoop] = 0;
		InitLoop += 1;
	}
}

///////////////////////////////////////////////////////////
// Mouse functions

// This function initializes the mouse
function Mouse_Init()
{
	mouse_map = yellowarrow;
	// turn the mouse on
	mouse_mode = 1;
	// keep updating the position of the mouse
	while (mouse_mode > 0)
	{
		mouse_pos.x = pointer.x;
		mouse_pos.y = pointer.y;
		wait(1);
	}
}

// This function returns the 3D mouse coordinates
// of where the mouse is hovering over the ground
function Mouse3D()
{
	var vector1;
	var vector2;
	
	vector1.x = mouse_pos.x;
	vector1.y = mouse_pos.y;
	vector1.z = 5;
	
	vector2.x = mouse_pos.x;
	vector2.y = mouse_pos.y;
	vector2.z = 5000;
	
	vec_for_screen(vector1, camera);
	vec_for_screen(vector2, camera);
	
	trace(vector1, vector2);
}

///////////////////////////////////////////////////////////
// Square.bmp actions (for testing Mouse3D())
// with snapping

// This function 'snaps' the position of the square
// over the nearest tile
function SnapIt()
{
	// x-axis snapping	
	if (my.x < 0)
	{
		my.x = TileSize * (int(my.x / TileSize)) - (TileSize / 2);
		if (my.x < -((GridLength / 2) * TileSize - (TileSize / 2)))
		{
			my.x = -((GridLength / 2) * TileSize - (TileSize / 2));
		}
	}
	if (my.x >= 0)
	{
		my.x = TileSize * (int(my.x / TileSize)) + (TileSize / 2);
		if (my.x > ((GridLength / 2) * TileSize - (TileSize / 2)))
		{
			my.x = ((GridLength / 2) * TileSize - (TileSize / 2));
		}
	}
	
	// y-axis snapping
	if (my.y < 0)
	{
		my.y = TileSize * (int(my.y / TileSize)) - (TileSize / 2);
		if (my.y < -((GridWidth / 2) * TileSize - (TileSize / 2)))
		{
			my.y = -((GridWidth / 2) * TileSize - (TileSize / 2));
		}
	}
	if (my.y >= 0)
	{
		my.y = TileSize * (int(my.y / TileSize)) + (TileSize / 2);
		if (my.y > ((GridWidth / 2) * TileSize - (TileSize / 2)))
		{
			my.y = ((GridWidth / 2) * TileSize - (TileSize / 2));
		}
	}
}

// This action controls the square.bmp sprite
action Square
{
	BlueSquare = me;
	
	my.passable = on;
	
	// set oriented mode on
	my.oriented = on;
	my.facing = off;
	my.decal = off;
	
	// set the orientation - up, of course
	my.pan = 0;
	my.roll = 0;
	my.tilt = 90;
	
	// keep updating the square with its position
	// underneath the mouse
	while(1)
	{
		Mouse3D();
		vec_set(my.x, target);
		my.z = 0.5;
		SnapIt();
		wait(1);
	}
}

///////////////////////////////////////////////////////////
// Path functions

// This function makes the node the blue square is over
// equal to 1
function NodeFill(num)
{
	var tempx;
	var tempY;
	
	tempx = (GridLength / 2) * TileSize - (TileSize / 2);
	tempy = (GridWidth / 2) * TileSize - (TileSize / 2);
	
	temp = ((BlueSquare.x + tempx) / TileSize) * GridWidth + ((BlueSquare.y + tempy) / TileSize);
	
	Nodes[temp] = num;
}

function NodeCheck()
{
	var tempx;
	var tempY;
	
	tempx = (GridLength / 2) * TileSize - (TileSize / 2);
	tempy = (GridWidth / 2) * TileSize - (TileSize / 2);
	
	temp = ((BlueSquare.x + tempx) / TileSize) * GridWidth + ((BlueSquare.y + tempy) / TileSize);
	
	return(Nodes[temp]);
}

// This action orients a path tile
action PathTile
{
	my.oriented = on;
	my.facing = off;
	my.decal = off;
	
	my.pan = 0;
	my.roll = 0;
	my.tilt = 90;
	
	my.z = 0.3;
	
	SnapIt();
}

// This function builds a path
function BuildPath()
{
	var vector1;
	var vector2;
	
	if (BuildMode == 0)
	{
		if (!NodeCheck())
		{
			Mouse3D();
			ent_create(earthtile, target, PathTile);
			NodeFill(1);
		}
		else
		{
			beep();
		}
	}
	else
	{
		if (NodeCheck())
		{
			vector1.x = BlueSquare.x;
			vector1.y = BlueSquare.y;
			vector1.z = 100;
			
			vector2.x = vector1.x;
			vector2.y = vector1.y;
			vector2.z = 0;
			
			trace_mode = ignore_passable;
			
			trace(vector1, vector2);
			ent_remove(you);
			
			NodeFill(0);
		}
		else
		{
			beep();
		}
	}
}

/*function BuildPath()
{
	Mouse3D();
	ent_create(earthtile, target, PathTile);
}*/

// This function toggles BuildMode to 1 for delete
function DeleteMode()
{
	if (BuildMode == 0)
	{
		BuildMode = 1;
		ent_morph(BlueSquare, square2);
	}
	else
	{
		BuildMode = 0;
		ent_morph(BlueSquare, square1);
	}
}

///////////////////////////////////////////////////////////
// On_* function declarations

on_click = BuildPath();
on_del = DeleteMode();